function [inputs]=scale(X, Y, kmax, init)
% Parameters: 
% X     - inputs
% Y     - desired outputs
% kmax  - [optional] maximum value for k for KNN-classification
% init  - [optional] initial values for weights, init should be n*n 
%         matrix containing weights for variables at its diagonal,
%         n is the number of variables. For example init = [2 0; 0 1]
%         puts initial weight for the first variable to 2 and for the 
%         second variable to 1

STOP       = 0.995; % stopping criteria
MIN_WEIGHT = 0;     % minimum weight for a variable
MAX_WEIGHT = 10;    % maximum weight for a variable

[lenght width] = size(X);

if (nargin < 3) % no kmax given
    kmax = 1;
end

if (nargin < 4) % no initial value for weights given
    weight = 5*eye(width); % initial weights: 5,5,5...5    
else
    weight = init;
end

% Hill climbing alrogithm
best_error = Inf;
current_weight = weight;
best_weight = 0;

stop = 0;
while (~stop)
    X2 = X*current_weight;    
    [error, k] = KNN(X2, Y, kmax);
    if (error <= best_error)
        if (error/best_error > STOP)
            stop = 1;
        end
        best_error = error;
        best_weight = current_weight;
    end
        
    % go throught neighbors of the current weight
    next_error = Inf;
    for j=1:width*2
        new_weight = current_weight;

        tmp_index = ceil(j/2); % index of the current weight
        
        % new value for the weight at index
        new_value = new_weight(tmp_index, tmp_index) + mod(j,2)*2-1; 

        % check upper and lower limits of the weights
        if (new_value <= MAX_WEIGHT && new_value >= MIN_WEIGHT)

            % update new weight
            new_weight(tmp_index, tmp_index) = new_value;

            X2 = X*new_weight;
            [error, k] = KNN(X2, Y, kmax);
            if (error < next_error)
                current_weight = new_weight;
                next_error = error;
            end
        end
    end
end

inputs = diag(best_weight);
