function [T, C, W] = pls(X,Y,a);
% Partial least squares regression projection to latent structures
% using NIPALS-algorithm
%
% >> [T, C, W, P]     = pls(X, Y, a)
% 
% Outputs    
%   T     : N x d matrix of X-scores (projected X:s)
%   C     : a x 1 projection matrix from latent space to output space (Yhat = T*C + F) 
%   W     : d x a projection matrix from input space to latent space (T = X*W)
% Inputs    
%   X     : N x d matrix with the inputs of the training data
%   Y     : N x 1 vector with the outputs of the training data
%   a     : Number of PLS-components to be extracted
% 

Xorig = X;
Yorig = Y;
[lenX, dimX] = size(X);
[lenY, dimY] = size(Y);
if lenX ~= lenY;
    error('Incompatible number of rows for X and Y');
end
if a > dimX;
    a = dimX;
    warning('The number of components excedes the dimension of X. Only dim(X) components extracted');
end

T=[];
C=[];
Ws=[];
P=[];
comp = 0;
epsilon = 10^-6;
while (comp<a),
    tmp = randperm(dimY);
    u = Y(:,tmp(1));
    dt = inf;
    tOld = inf;
    while (dt>epsilon),
        ws = X'*u/(u'*u);
        ws = ws/norm(ws);
        t = X*ws;
        c = Y'*t/(t'*t);
        u = Y*c/(c'*c);
        dt = norm(t-tOld)/norm(t);
        tOld = t;
    end
    % deflation
    p = X'*t/(t'*t);
    X = X-t*p';
    Y = Y-t*c;
    % output variables
    T = [T,t];
    C = [C;c];
    Ws = [Ws,ws];
    P = [P,p];
    comp = comp+1;
end

W = Ws*inv(P'*Ws);
