function [variables, beta, Cp] = LARS(X, y)

% Least angle regression
%
% variables = LARS(X, y) calculates Least angle regression on input
% variables X and output variable y. Returns the order in which the
% variables were selected by LARS. X is in NxM matrix with N
% observations and M variables. y is Nx1 vector.
%
% [variables, beta, Cp] = LARS(X, y) returns also coefficients in each
% step as columns of beta and risk estimation values Cp. After
% selecting first i variables (whose indexes are variables(1:i)) the
% prediction of the model is beta(:, i)*X. If the data comes from
% normal distribution with mean mu, the expected normalized deviation
% between true mu and model's prediction is minimized when Cp
% attains its minimal value.

% Antti Ajanki, 14.11.2006

% scale the variables
[N, M] = size(X); % N observations, M variables
meany=mean(y);
y = y(:)-meany;
meanx = mean(X);
X = X-repmat(meanx, N, 1);
scalex = sqrt(sum(X.^2));
X = X./repmat(scalex, N, 1);

% find largest absolute correlation
mu = zeros(N, 1);
c = X'*(y-mu);
[maxcor, variables] = max(abs(c));

r1 = zeros(M, 1);
r2 = zeros(M, 1);
beta = zeros(M, M+1);
tol = 1e-14;
k = 1;
while k <= M
  % find equiangular direction
  c = X'*(y-mu);
  s = sign(c(variables))';
  XA = X(:, variables).*repmat(s, N, 1);
  invGA = inv(XA'*XA);
  AA = (sum(sum(invGA)))^(-1/2);  
  w = AA*sum(invGA, 2);
  equiangvect = XA*w;

  % compute step length and find next variable
  if k==M
    % In the last stage take full step
    gamma = maxcor/AA;
  else
    a = X'*equiangvect;

    z = abs(a-AA) > eps;
    r1(z) = (maxcor-c(z))./(AA-a(z));
    r1(~z) = Inf;
    z = abs(AA+a) > eps;
    r2(z) = (maxcor+c(z))./(AA+a(z));
    r2(~z) = Inf;

    r1(r1 < 0) = Inf;
    r1(variables) = Inf;
    r2(r2 < 0) = Inf;
    r2(variables) = Inf;
    [gamma, i] = min([r1; r2]);
    if i > M
      i = i-M;
    end
  end

  % update coefficients
  beta(variables, k+1) = beta(variables, k) + gamma*s'.*w;
  k = k+1;
  
  maxcor = maxcor-gamma*AA;
  mu = mu + gamma*equiangvect;
  variables = [variables i];
end

% leave out extra columns
beta = beta(:, 2:end);
variables = variables(1:end-1);

% compute Cp
residuals = repmat(y, 1, M)-X*beta;
RSS = sum(residuals.^2);
Cp = ((N-M-1)*RSS)/RSS(end) - N + 2*(1:M);

% rescale
beta = beta.*repmat(1./scalex', 1, M);
