function [inputs,costs,pop,cost,parG,trace] = GSearch(X,Y,costfunction,parC,parG)
% GSearch  Genetic algorithm for input selection
%
% inputs = GSearch(X,Y,costfunction);
% [inputs,costs,pop,pcosts,parG,trace] = GSearch(X,Y,costfunction,parC,parG)
%
% Examples:
%
%   inputs = GSearch(X,Y,@KNN);
%
%   parG.verbose = true;
%   inputs = GSearch(X,Y,@KNN,[],parG);
%
%   [inputs, costs, pop, pcosts, parG, trace] = GSearch(X,Y,@KNN);
%
% Parameters:
%   X is a Nxd matrix of d-dimensional inputs samples
%   Y is a Nx1 vector of corresponding output samples
%   costfunction is a function with the interface
%      [error]=costfunction(X, Y [,parC])
%   parC is the parameters for costfunction
%   parG is a struct of parameters with default values
%      doInit: true                  % Create initial population with
%      initFun: @GSBinInitialize     % par.initPop=fun(X,Y,parG)
%      popSize: 50                   % Population size 
%      elitPopSize: 2                % includes elite population size
%      selFun: @GSRouletteSelection  % inds=fun(pop,cost,gen,parG)
%      reprodFuns: {@GSBinMutation @GSBinXover} % pop=fun(pop,cost,gen,parR)
%      reprodPars: {1/d 0.5}         % arguments parR for above
%      reprodProbs: [0.5 0.5]        % Probs for reprodFuns 
%      termFun: @GSMaxGenTermination % tf=fun(pop,cost,gen,parG)
%      termPar: 200                  % with function special parameter
%      verbose: false                % Print some statistics
%
% Returns:
%   inputs is a vector of selected input indices
%   costs is a vector of min cost for each generation
%   pop is the final population 
%   pcosts is a vector of the costs corresponding to the final population
%   parG is the augmented parameter structure
%   trace is a trace of populations and cost for each generation
%
% Genetic algorithm works as follows
% 
%   P <- Initial population
%   C <- Costs for individuals in P
%   G <- 1
%   do:
%      S  <- Select individuals for reproduction  
%      P' <- Apply genetic operators (crossover and mutation) to S
%      P  <- Elite of P and P'
%      G  <- G + 1
%   while not Terminated
%   Return the individual with lowest cost
  
% Copyright 
% (C) 2006 Jaakko Vyrynen <jaakko.j.vayrynen@tkk.fi>
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  % Check input parameters
  if nargin < 3, 
    error('X,Y and costfunction must be defined.'); 
  end
  if nargin < 4, parC = []; end % Parameters for cost function 
  if nargin < 5, parG = []; end % Parameters for genetic algorithm

  % Some checks for input parameter types
  if ndims(X)~=2 || ndims(Y)~=2 || size(X,1)~=size(Y,1) || size(Y,2)~=1,
    error('X should be a Nxd matrix and Y a Nx1 vector.')
  end
  if ~isa(costfunction, 'function_handle'),
    error('costfunction should be a function handle');
  end

  % Set default parameters if not set
  defParG=struct('doInit', true, ...
		 'initFun',@GSBinInitialize, ...
		 'popSize', 50, ...
		 'elitPopSize', 2, ... 
		 'selFun',@GSRouletteSelection, ...
		 'reprodFuns', {{@GSBinMutation @GSBinXover}}, ...
		 'reprodProbs', [0.5 0.5], ...
		 'reprodPars', {{1/size(X,2) 0.5}}, ...
		 'termFun', @GSMaxGenTermination, ...
		 'termPar', 200, ...
		 'verbose', false);

  defFields = fieldnames(defParG);
  for k = 1:length(defFields),
    field = defFields{k};
    if ~isfield(parG, field),
      parG.(field) = defParG.(field);
    end
  end

  % Cumulative probabilites for reproduction functions 
  reprodCumProbs = [0 cumsum(parG.reprodProbs)];
  if sum(parG.reprodProbs) ~= 1,
    warning('Reproduction probabilities do not sum to 1.');
  end
  
  % Initialize population for first generation or use given
  if parG.doInit,
    parG.initPop = parG.initFun(X, Y, parG);
  end
  if parG.popSize ~= size(parG.initPop, 1),
    error('Actual population size and value of par.popSize do not match.');
  end
  gen = 1; % Generation number
  pop = parG.initPop; % Current population [PopSize x d] logical matrix
  
  % Calculate costs for initial population and sort.
  % This is slow for KNN and lots of samples
  cost = zeros(1, parG.popSize);
  for k = 1:parG.popSize,
    if any(pop(k,:)),
      cost(k) = costfunction(X(:,pop(k,:)), Y, parC);
    else,
      cost(k) = Inf;
    end
  end
  [cost order] = sort(cost);
  pop = pop(order, :);
  best = pop(1,:);
  minCost = cost(1);
  costs = cost(1);
  
  % Repeat until termination condition is true
  done = false;
  while (~done),

    % Verbose information about progress
    if parG.verbose,
      fprintf('gen=%d, bestCost=%g, best=', gen, minCost);
      fprintf('%d',best);
      fprintf('\n');
    end
    
    % Trace of population and costs 
    if nargout >= 4,
      trace.pop{gen} = pop;
      trace.cost{gen} = cost;
    end

    % Create children based on elite and reproduction
    newPop = logical(zeros(size(pop)));
    newPopFill = 0; % Number of individuals so far

    % Elitist selection
    newPop(1:parG.elitPopSize, :) = pop(1:parG.elitPopSize, :);
    newPopFill = newPopFill + parG.elitPopSize;

    % Select individuals for reproduction
    sel = parG.selFun(pop, cost, gen, parG);
    sel = sel(1:(parG.popSize-newPopFill));
    
    % Apply reproduction functions to selected individuals 
    % according to the probabilities of applying functions 
    r = rand(size(sel));
    for i = 1:length(parG.reprodFuns),
      b = r>=reprodCumProbs(i) & r<reprodCumProbs(i+1);
      if any(b),
	Z = parG.reprodFuns{i}(pop(b,:), cost, gen, parG.reprodPars{i});
	newPop(newPopFill+(1:sum(b)),:) = Z;
	newPopFill = newPopFill + sum(b);
      end
    end

    % Accept population and move to next generation
    pop = newPop;
    gen = gen + 1;

    % Evaluate costs for new generation and sort.
    % This is slow for KNN and lots of samples
    cost = zeros(1, parG.popSize);
    for k = 1:parG.popSize,
      if any(pop(k,:)),
	cost(k) = costfunction(X(:,pop(k,:)), Y, parC);
      else,
	cost(k) = Inf;
      end
    end
    [cost order] = sort(cost);
    pop = pop(order, :);
    costs(end+1) = cost(1);
    if cost(1) < minCost, % Remember best seen solution without elite
      best = pop(1,:);
      minCost = cost(1);
    end
    
    % Check termination criteria
    done = parG.termFun(pop, cost, gen, parG);

  end
  
  % Return best seen solution 
  inputs = find(best);
