function [error] = Boot632_KNN(x, y, varargin)
%
% Bootstrap632 with KNN
% 
% Params:
% x    The data as a Nxd matrix. d is the dimension of the input space.
% y    The outputs as a Nx1 matrix.
%
% Optional:
% var.J     Amount of Bootstrap 632 iterations
%           Default is 100.
% var.neigh The maximum k.
%           Default is 10% of the size of the data set.
% 
% Return:
% error  The Bootstrap error.
% k      The selected k.
%
% Examples:
% [error] = Boot632_KNN(x, y); % J = 100, neigh = floor(rows/10);
% [error] = Boot632_KNN(x, y, struct('J', 50, 'neigh', 13)); 
% [error] = Boot632_KNN(x, y, struct('J', 50)); %  neigh = floor(rows/10);
% [error] = Boot632_KNN(x, y, struct('neigh', 13)); % J = 20
%

[rows, columns] = size(x);

if nargin == 3
   varargin = varargin{1};
   if(isfield(varargin, 'J'))
     J = getfield(varargin, 'J');
   else J = 100; % default value for J
   end;
   if(isfield(varargin, 'neigh'))
     neigh = getfield(varargin, 'neigh');
   else neigh = floor(rows/10); % default value neigh = floor(rows/10);
   end;
   if neigh > rows
      neigh = rows - 1; 
   end;
   else if nargin == 2
        J = 100; % default value for J
        neigh = floor(rows/10); % default value neigh
   end;
end;

if max(size(y)) ~= rows

  disp('Output must have as many rows as the input data!');

  error = inf;

  k = inf;

  return;

end

E_I_I = KNN(x,y,neigh); 

% Make J Bootstrap iterations

for j = 1 : J

  % Take randomly samples from the original data:
  boot=ceil(rand(rows,1)*rows);
  
  % Take samples that are not in the Bootstrap set:
  unsel = setdiff([1:rows]', boot);
  xunsel = x(unsel,:); yunsel = y(unsel,:);
  % Let's teach the model with unselected points and get k:
  [error,k] = KNN(xunsel, yunsel, neigh);

  % Count the error in the bootstrap set using the selected k:
  xboot = x(boot,:); yboot = y(boot,:);
  opt632(j) = knn(xboot,yboot, k); % knn  -method at the end of the file

end;

% Calculate the generalization error:
error = 0.368*mean(opt632)+0.632*E_I_I;


function [error] = knn(x, y, k)

% KNN part of the method: get error using k-nearest neighbours
% 
% Params:
% x    The data as a Nxd matrix. d is the dimension of the input space.
% y    The outputs as a Nx1 matrix.
%
%Optional:
% k    Value of k to be   
% 
%Return:
% error Error of using k-nearest neighbours

[rows, columns] = size(x);

% index(k+1,i): index to kth nearest neighbour of sample i 
[sorted, index] = sort(squareform(pdist(x, 'euclidean')));

if k == 1

  [error, k] = min(mean((y(index(2, :)) - y) .^2, 1));

else

% yhat(k,i): average of y of values of sample i and k nearest neighbours
yhat = (cumsum(y(index(2:k+1, :)),1)' ./ (ones(rows,1) * (1:k)));
% Count error of the whole set for k:
errors = mean((yhat - y * ones(1,k)) .^2);
% error of k-nearest-neighbours:
error = errors(k);

end;
