function out = dt_t1_plotsignal(in, Fs, thr)
% DT_T1_PLOTSIGNAL plots signal, computes
% energy in 50 ms windows and removes silent
% parts in the beginning and end of signal
%
%   out = dt_t1_plotsignal(in, Fs)
%
%   in     (vector) input signal
%   Fs     (scalar) sampling frequency
%   thr    (scalar) threshold for energy
%   out    (vector) modified signal

% Jukka Parviainen, 9.11.2006

t1 = [1 : length(in)]/Fs;

% plotting top figure
subplot(3,1,1);  % 3 rows, 1 column, 1st graph
plot(t1, in);
grid on;
title(['Input signal ' inputname(1) ' (' num2str(t1(end)) ' s)']);
xlabel('time (s)');

% computing energy
n50    = round(0.05 * Fs);     % 50 ms == n50 samples
N      = ceil(length(in)/n50); % number of 50ms windows
Energy = zeros(N-1,1);

for k = [1 : N-1]
    Energy(k) = sum(in((k-1)*n50+1: k*n50).^2);
end;

% plotting energy
subplot(3,1,2);  % 3 rows, 1 column, 2nd graph
bar(n50*[1:N-1]/Fs, Energy);
grid on;
title(['Energy of input signal ' inputname(1)]);
li = line([t1(1) t1(end)],[thr thr]);
set(li, 'Color', [1 0 0]);
xlabel('time (s)');
ylabel('energy');

% removing silent parts from both sides of signal
indices = find(Energy > thr);
signalStarts = (min(indices)-2)*n50
signalEnds = (max(indices)+1)*n50

if (signalStarts < 1)
    signalStarts = 1;
end;
if ((signalEnds > length(in)) | (signalEnds < signalStarts))
    signalEnds = length(in);
end;

out = in(signalStarts : signalEnds);

% plot the output signal
t2 = [1 : length(out)]/Fs;
subplot(3,1,3);  % 3 rows, 1 column, 3rd graph
plot(t2, out);
grid on;
title(['Output signal  (' num2str(t2(end)) ' s)']);
xlabel('time (s)');




